# -*- coding: utf-8 -*- #
# Copyright 2024 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Flags for Mirroring Endpoint Group commands."""


from googlecloudsdk.api_lib.network_security.mirroring_endpoint_groups import api
from googlecloudsdk.calliope import arg_parsers
from googlecloudsdk.calliope import base
from googlecloudsdk.calliope import parser_arguments
from googlecloudsdk.calliope.concepts import concepts
from googlecloudsdk.calliope.concepts import deps
from googlecloudsdk.command_lib.util.concepts import concept_parsers
from googlecloudsdk.command_lib.util.concepts import presentation_specs
from googlecloudsdk.core import properties
from googlecloudsdk.core import resources


ENDPOINT_GROUP_RESOURCE_NAME = "MIRRORING_ENDPOINT_GROUP"
ENDPOINT_GROUP_RESOURCE_COLLECTION = (
    "networksecurity.projects.locations.mirroringEndpointGroups"
)
DEPLOYMENT_GROUP_RESOURCE_COLLECTION = (
    "networksecurity.projects.locations.mirroringDeploymentGroups"
)

_PACKET_BROKER_SUPPORTED = (base.ReleaseTrack.ALPHA,)


def AddEndpointGroupResource(release_track, parser):
  """Adds Mirroring Endpoint Group resource."""
  api_version = api.GetApiVersion(release_track)
  resource_spec = concepts.ResourceSpec(
      ENDPOINT_GROUP_RESOURCE_COLLECTION,
      "mirroring endpoint group",
      api_version=api_version,
      projectsId=concepts.DEFAULT_PROJECT_ATTRIBUTE_CONFIG,
      locationsId=concepts.ResourceParameterAttributeConfig(
          "location",
          "Location of the {resource}.",
          parameter_name="locationsId",
      ),
      mirroringEndpointGroupsId=concepts.ResourceParameterAttributeConfig(
          "endpoint-group-id",
          "Id of the {resource}",
          parameter_name="mirroringEndpointGroupsId",
      ),
  )
  presentation_spec = presentation_specs.ResourcePresentationSpec(
      name=ENDPOINT_GROUP_RESOURCE_NAME,
      concept_spec=resource_spec,
      required=True,
      group_help="Mirroring Endpoint Group.",
  )
  return concept_parsers.ConceptParser([presentation_spec]).AddToParser(parser)


def AddMaxWait(
    parser,
    default_max_wait,
    help_text="Time to synchronously wait for the operation to complete, after which the operation continues asynchronously. Ignored if --no-async isn't specified. See $ gcloud topic datetimes for information on time formats.",
):
  """Adds --max-wait flag."""
  parser.add_argument(
      "--max-wait",
      dest="max_wait",
      required=False,
      default=default_max_wait,
      help=help_text,
      type=arg_parsers.Duration(),
  )


def MakeGetUriFunc(release_track):
  return lambda x: api.GetEffectiveApiEndpoint(release_track) + x.name


def LocationAttributeConfig(default="global"):
  """Gets Google Cloud location resource attribute."""
  fallthroughs = []
  if default:
    fallthroughs.append(
        deps.Fallthrough(
            lambda: default,
            "Location of the Mirroring Endpoint Group. Defaults to {}".format(
                default
            ),
        )
    )
  return concepts.ResourceParameterAttributeConfig(
      name="location",
      help_text="Location of the {resource}.",
      fallthroughs=fallthroughs,
  )


def GetLocationResourceSpec():
  """Constructs and returns the Resource specification for Location."""
  return concepts.ResourceSpec(
      "networksecurity.projects.locations",
      resource_name="location",
      locationsId=LocationAttributeConfig(),
      projectsId=concepts.DEFAULT_PROJECT_ATTRIBUTE_CONFIG,
  )


def AddLocationResourceArg(
    parser: parser_arguments.ArgumentInterceptor, help_text
):
  """Adds a resource argument for Google Cloud location.

  Args:
    parser: The argparse.parser to add the resource arg to.
    help_text: str, the text of the help message.
  """
  concept_parsers.ConceptParser.ForResource(
      "--location",
      GetLocationResourceSpec(),
      help_text,
      required=True,
  ).AddToParser(parser)


def AddDescriptionArg(parser, help_text="Description of the endpoint"):
  """Adds a resource argument for Google Cloud description."""
  parser.add_argument("--description", required=False, help=help_text)


def AddMirroringDeploymentGroupResource(release_track, parser):
  """Adds mirroring deployment group resource."""
  api_version = api.GetApiVersion(release_track)
  collection_info = resources.REGISTRY.Clone().GetCollectionInfo(
      ENDPOINT_GROUP_RESOURCE_COLLECTION, api_version
  )
  resource_spec = concepts.ResourceSpec(
      DEPLOYMENT_GROUP_RESOURCE_COLLECTION,
      "mirroring deployment group",
      api_version=api_version,
      projectsId=concepts.ResourceParameterAttributeConfig(
          name="project",
          help_text="Project of the {resource}.",
          parameter_name="projectsId",
          fallthroughs=[
              deps.ArgFallthrough("--project"),
              deps.PropertyFallthrough(properties.VALUES.core.project),
              deps.FullySpecifiedAnchorFallthrough(
                  [deps.ArgFallthrough(ENDPOINT_GROUP_RESOURCE_NAME)],
                  collection_info,
                  "projectsId",
              ),
          ],
      ),
      locationsId=concepts.ResourceParameterAttributeConfig(
          "location",
          "Location of the {resource}.",
          parameter_name="locationsId",
          fallthroughs=[
              deps.ArgFallthrough("--location"),
              deps.FullySpecifiedAnchorFallthrough(
                  [deps.ArgFallthrough(ENDPOINT_GROUP_RESOURCE_NAME)],
                  collection_info,
                  "locationsId",
              ),
          ],
      ),
      mirroringDeploymentGroupsId=concepts.ResourceParameterAttributeConfig(
          "id",
          "Id of the {resource}",
          parameter_name="mirroringDeploymentGroupsId",
      ),
  )
  presentation_spec = presentation_specs.ResourcePresentationSpec(
      name="--mirroring-deployment-group",
      concept_spec=resource_spec,
      required=False,
      group_help="Mirroring Deployment Group.",
      flag_name_overrides={"project": "--mirroring-deployment-group-project"},
      prefixes=True,
  )
  concept_parsers.ConceptParser([presentation_spec]).AddToParser(parser)


def AddDeploymentGroupMutexGroup(release_track, parser):
  """Adds mirroring deployment groups resource."""
  mutex_group = parser.add_group(mutex=True, required=True)

  # This flag is added to the mutex group for ALL release tracks.
  AddMirroringDeploymentGroupResource(release_track, mutex_group)

  # This plural flag is ONLY added to the mutex group for the BROKER-supported
  # tracks.
  if release_track in _PACKET_BROKER_SUPPORTED:
    mutex_group.add_argument(
        "--mirroring-deployment-groups",
        metavar="MIRRORING_DEPLOYMENT_GROUPS",
        type=arg_parsers.ArgList(min_length=1),
        help=(
            "A comma-separated list of Mirroring Deployment Groups to associate"
            " with the Endpoint Group. Each deployment group must be specified"
            " by its full resource name. e.g."
            " --mirroring-deployment-groups=projects/123/locations/global/mirroringDeploymentGroups/my-dg1,projects/456/locations/global/mirroringDeploymentGroups/my-dg2"
        ),
        required=False,
    )


# The type flag is only supported for Alpha. When not specified, the default
# type is DIRECT.
def AddType(parser):
  """Adds the endpoint group type flag to the parser."""
  parser.add_argument(
      "--type",
      choices=["DIRECT", "BROKER"],
      help=(
          "The type of the endpoint group. If not specified, the default type"
          " is DIRECT."
      ),
  )
