/*	$OpenBSD: sha_test.c,v 1.6 2023/07/19 15:11:42 joshua Exp $ */
/*
 * Copyright (c) 2022, 2023 Joshua Sing <joshua@hypera.dev>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <openssl/evp.h>
#include <openssl/sha.h>

#include <stdint.h>
#include <string.h>

struct sha_test {
	const int algorithm;
	const uint8_t in[128];
	const size_t in_len;
	const uint8_t out[EVP_MAX_MD_SIZE];
};

static const struct sha_test sha_tests[] = {
	/* SHA-1 */
	{
		.algorithm = NID_sha1,
		.in = "abc",
		.in_len = 3,
		.out = {
			0xa9, 0x99, 0x3e, 0x36, 0x47, 0x06, 0x81, 0x6a,
			0xba, 0x3e, 0x25, 0x71, 0x78, 0x50, 0xc2, 0x6c,
			0x9c, 0xd0, 0xd8, 0x9d,
		},
	},
	{
		.algorithm = NID_sha1,
		.in = "",
		.in_len = 0,
		.out = {
			0xda, 0x39, 0xa3, 0xee, 0x5e, 0x6b, 0x4b, 0x0d,
			0x32, 0x55, 0xbf, 0xef, 0x95, 0x60, 0x18, 0x90,
			0xaf, 0xd8, 0x07, 0x09,
		},
	},
	{
		.algorithm = NID_sha1,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x84, 0x98, 0x3e, 0x44, 0x1c, 0x3b, 0xd2, 0x6e,
			0xba, 0xae, 0x4a, 0xa1, 0xf9, 0x51, 0x29, 0xe5,
			0xe5, 0x46, 0x70, 0xf1,
		},
	},
	{
		.algorithm = NID_sha1,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0xa4, 0x9b, 0x24, 0x46, 0xa0, 0x2c, 0x64, 0x5b,
			0xf4, 0x19, 0xf9, 0x95, 0xb6, 0x70, 0x91, 0x25,
			0x3a, 0x04, 0xa2, 0x59,
		},
	},

	/* SHA-224 */
	{
		.algorithm = NID_sha224,
		.in = "abc",
		.in_len = 3,
		.out = {
			0x23, 0x09, 0x7d, 0x22, 0x34, 0x05, 0xd8, 0x22,
			0x86, 0x42, 0xa4, 0x77, 0xbd, 0xa2, 0x55, 0xb3,
			0x2a, 0xad, 0xbc, 0xe4, 0xbd, 0xa0, 0xb3, 0xf7,
			0xe3, 0x6c, 0x9d, 0xa7,
		},
	},
	{
		.algorithm = NID_sha224,
		.in = "",
		.in_len = 0,
		.out = {
			0xd1, 0x4a, 0x02, 0x8c, 0x2a, 0x3a, 0x2b, 0xc9,
			0x47, 0x61, 0x02, 0xbb, 0x28, 0x82, 0x34, 0xc4,
			0x15, 0xa2, 0xb0, 0x1f, 0x82, 0x8e, 0xa6, 0x2a,
			0xc5, 0xb3, 0xe4, 0x2f,
		},
	},
	{
		.algorithm = NID_sha224,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x75, 0x38, 0x8b, 0x16, 0x51, 0x27, 0x76, 0xcc,
			0x5d, 0xba, 0x5d, 0xa1, 0xfd, 0x89, 0x01, 0x50,
			0xb0, 0xc6, 0x45, 0x5c, 0xb4, 0xf5, 0x8b, 0x19,
			0x52, 0x52, 0x25, 0x25,
		},
	},
	{
		.algorithm = NID_sha224,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0xc9, 0x7c, 0xa9, 0xa5, 0x59, 0x85, 0x0c, 0xe9,
			0x7a, 0x04, 0xa9, 0x6d, 0xef, 0x6d, 0x99, 0xa9,
			0xe0, 0xe0, 0xe2, 0xab, 0x14, 0xe6, 0xb8, 0xdf,
			0x26, 0x5f, 0xc0, 0xb3,
		},
	},

	/* SHA-256 */
	{
		.algorithm = NID_sha256,
		.in = "abc",
		.in_len = 3,
		.out = {
			0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
			0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
			0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
			0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad,
		},
	},
	{
		.algorithm = NID_sha256,
		.in = "",
		.in_len = 0,
		.out = {
			0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14,
			0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24,
			0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c,
			0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55,
		},
	},
	{
		.algorithm = NID_sha256,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
			0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
			0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
			0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1,
		},
	},
	{
		.algorithm = NID_sha256,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0xcf, 0x5b, 0x16, 0xa7, 0x78, 0xaf, 0x83, 0x80,
			0x03, 0x6c, 0xe5, 0x9e, 0x7b, 0x04, 0x92, 0x37,
			0x0b, 0x24, 0x9b, 0x11, 0xe8, 0xf0, 0x7a, 0x51,
			0xaf, 0xac, 0x45, 0x03, 0x7a, 0xfe, 0xe9, 0xd1,
		},
	},

	/* SHA-384 */
	{
		.algorithm = NID_sha384,
		.in = "abc",
		.in_len = 3,
		.out = {
			0xcb, 0x00, 0x75, 0x3f, 0x45, 0xa3, 0x5e, 0x8b,
			0xb5, 0xa0, 0x3d, 0x69, 0x9a, 0xc6, 0x50, 0x07,
			0x27, 0x2c, 0x32, 0xab, 0x0e, 0xde, 0xd1, 0x63,
			0x1a, 0x8b, 0x60, 0x5a, 0x43, 0xff, 0x5b, 0xed,
			0x80, 0x86, 0x07, 0x2b, 0xa1, 0xe7, 0xcc, 0x23,
			0x58, 0xba, 0xec, 0xa1, 0x34, 0xc8, 0x25, 0xa7,
		},
	},
	{
		.algorithm = NID_sha384,
		.in = "",
		.in_len = 0,
		.out = {
			0x38, 0xb0, 0x60, 0xa7, 0x51, 0xac, 0x96, 0x38,
			0x4c, 0xd9, 0x32, 0x7e, 0xb1, 0xb1, 0xe3, 0x6a,
			0x21, 0xfd, 0xb7, 0x11, 0x14, 0xbe, 0x07, 0x43,
			0x4c, 0x0c, 0xc7, 0xbf, 0x63, 0xf6, 0xe1, 0xda,
			0x27, 0x4e, 0xde, 0xbf, 0xe7, 0x6f, 0x65, 0xfb,
			0xd5, 0x1a, 0xd2, 0xf1, 0x48, 0x98, 0xb9, 0x5b,
		},
	},
	{
		.algorithm = NID_sha384,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x33, 0x91, 0xfd, 0xdd, 0xfc, 0x8d, 0xc7, 0x39,
			0x37, 0x07, 0xa6, 0x5b, 0x1b, 0x47, 0x09, 0x39,
			0x7c, 0xf8, 0xb1, 0xd1, 0x62, 0xaf, 0x05, 0xab,
			0xfe, 0x8f, 0x45, 0x0d, 0xe5, 0xf3, 0x6b, 0xc6,
			0xb0, 0x45, 0x5a, 0x85, 0x20, 0xbc, 0x4e, 0x6f,
			0x5f, 0xe9, 0x5b, 0x1f, 0xe3, 0xc8, 0x45, 0x2b,
		},
	},
	{
		.algorithm = NID_sha384,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
			0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
			0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
			0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
			0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
			0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39,
		},
	},

	/* SHA-512 */
	{
		.algorithm = NID_sha512,
		.in = "abc",
		.in_len = 3,
		.out = {
			0xdd, 0xaf, 0x35, 0xa1, 0x93, 0x61, 0x7a, 0xba,
			0xcc, 0x41, 0x73, 0x49, 0xae, 0x20, 0x41, 0x31,
			0x12, 0xe6, 0xfa, 0x4e, 0x89, 0xa9, 0x7e, 0xa2,
			0x0a, 0x9e, 0xee, 0xe6, 0x4b, 0x55, 0xd3, 0x9a,
			0x21, 0x92, 0x99, 0x2a, 0x27, 0x4f, 0xc1, 0xa8,
			0x36, 0xba, 0x3c, 0x23, 0xa3, 0xfe, 0xeb, 0xbd,
			0x45, 0x4d, 0x44, 0x23, 0x64, 0x3c, 0xe8, 0x0e,
			0x2a, 0x9a, 0xc9, 0x4f, 0xa5, 0x4c, 0xa4, 0x9f,
		},
	},
	{
		.algorithm = NID_sha512,
		.in = "",
		.in_len = 0,
		.out = {
			0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd,
			0xf1, 0x54, 0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07,
			0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc,
			0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce,
			0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
			0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f,
			0x63, 0xb9, 0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81,
			0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e,
		},
	},
	{
		.algorithm = NID_sha512,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x20, 0x4a, 0x8f, 0xc6, 0xdd, 0xa8, 0x2f, 0x0a,
			0x0c, 0xed, 0x7b, 0xeb, 0x8e, 0x08, 0xa4, 0x16,
			0x57, 0xc1, 0x6e, 0xf4, 0x68, 0xb2, 0x28, 0xa8,
			0x27, 0x9b, 0xe3, 0x31, 0xa7, 0x03, 0xc3, 0x35,
			0x96, 0xfd, 0x15, 0xc1, 0x3b, 0x1b, 0x07, 0xf9,
			0xaa, 0x1d, 0x3b, 0xea, 0x57, 0x78, 0x9c, 0xa0,
			0x31, 0xad, 0x85, 0xc7, 0xa7, 0x1d, 0xd7, 0x03,
			0x54, 0xec, 0x63, 0x12, 0x38, 0xca, 0x34, 0x45,
		},
	},
	{
		.algorithm = NID_sha512,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
			0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
			0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
			0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
			0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
			0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
			0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
			0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09,
		},
	},

	/* SHA3-224 */
	{
		.algorithm = NID_sha3_224,
		.in = "abc",
		.in_len = 3,
		.out = {
			0xe6, 0x42, 0x82, 0x4c, 0x3f, 0x8c, 0xf2, 0x4a,
			0xd0, 0x92, 0x34, 0xee, 0x7d, 0x3c, 0x76, 0x6f,
			0xc9, 0xa3, 0xa5, 0x16, 0x8d, 0x0c, 0x94, 0xad,
			0x73, 0xb4, 0x6f, 0xdf,
		},
	},
	{
		.algorithm = NID_sha3_224,
		.in = "",
		.in_len = 0,
		.out = {
			0x6b, 0x4e, 0x03, 0x42, 0x36, 0x67, 0xdb, 0xb7,
			0x3b, 0x6e, 0x15, 0x45, 0x4f, 0x0e, 0xb1, 0xab,
			0xd4, 0x59, 0x7f, 0x9a, 0x1b, 0x07, 0x8e, 0x3f,
			0x5b, 0x5a, 0x6b, 0xc7,
		},
	},
	{
		.algorithm = NID_sha3_224,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x8a, 0x24, 0x10, 0x8b, 0x15, 0x4a, 0xda, 0x21,
			0xc9, 0xfd, 0x55, 0x74, 0x49, 0x44, 0x79, 0xba,
			0x5c, 0x7e, 0x7a, 0xb7, 0x6e, 0xf2, 0x64, 0xea,
			0xd0, 0xfc, 0xce, 0x33,
		},
	},
	{
		.algorithm = NID_sha3_224,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0x54, 0x3e, 0x68, 0x68, 0xe1, 0x66, 0x6c, 0x1a,
			0x64, 0x36, 0x30, 0xdf, 0x77, 0x36, 0x7a, 0xe5,
			0xa6, 0x2a, 0x85, 0x07, 0x0a, 0x51, 0xc1, 0x4c,
			0xbf, 0x66, 0x5c, 0xbc,
		},
	},

	/* SHA3-256 */
	{
		.algorithm = NID_sha3_256,
		.in = "abc",
		.in_len = 3,
		.out = {
			0x3a, 0x98, 0x5d, 0xa7, 0x4f, 0xe2, 0x25, 0xb2,
			0x04, 0x5c, 0x17, 0x2d, 0x6b, 0xd3, 0x90, 0xbd,
			0x85, 0x5f, 0x08, 0x6e, 0x3e, 0x9d, 0x52, 0x5b,
			0x46, 0xbf, 0xe2, 0x45, 0x11, 0x43, 0x15, 0x32,
		},
	},
	{
		.algorithm = NID_sha3_256,
		.in = "",
		.in_len = 0,
		.out = {
			0xa7, 0xff, 0xc6, 0xf8, 0xbf, 0x1e, 0xd7, 0x66,
			0x51, 0xc1, 0x47, 0x56, 0xa0, 0x61, 0xd6, 0x62,
			0xf5, 0x80, 0xff, 0x4d, 0xe4, 0x3b, 0x49, 0xfa,
			0x82, 0xd8, 0x0a, 0x4b, 0x80, 0xf8, 0x43, 0x4a,
		},
	},
	{
		.algorithm = NID_sha3_256,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x41, 0xc0, 0xdb, 0xa2, 0xa9, 0xd6, 0x24, 0x08,
			0x49, 0x10, 0x03, 0x76, 0xa8, 0x23, 0x5e, 0x2c,
			0x82, 0xe1, 0xb9, 0x99, 0x8a, 0x99, 0x9e, 0x21,
			0xdb, 0x32, 0xdd, 0x97, 0x49, 0x6d, 0x33, 0x76,
		},
	},
	{
		.algorithm = NID_sha3_256,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0x91, 0x6f, 0x60, 0x61, 0xfe, 0x87, 0x97, 0x41,
			0xca, 0x64, 0x69, 0xb4, 0x39, 0x71, 0xdf, 0xdb,
			0x28, 0xb1, 0xa3, 0x2d, 0xc3, 0x6c, 0xb3, 0x25,
			0x4e, 0x81, 0x2b, 0xe2, 0x7a, 0xad, 0x1d, 0x18,
		},
	},

	/* SHA3-384 */
	{
		.algorithm = NID_sha3_384,
		.in = "abc",
		.in_len = 3,
		.out = {
			0xec, 0x01, 0x49, 0x82, 0x88, 0x51, 0x6f, 0xc9,
			0x26, 0x45, 0x9f, 0x58, 0xe2, 0xc6, 0xad, 0x8d,
			0xf9, 0xb4, 0x73, 0xcb, 0x0f, 0xc0, 0x8c, 0x25,
			0x96, 0xda, 0x7c, 0xf0, 0xe4, 0x9b, 0xe4, 0xb2,
			0x98, 0xd8, 0x8c, 0xea, 0x92, 0x7a, 0xc7, 0xf5,
			0x39, 0xf1, 0xed, 0xf2, 0x28, 0x37, 0x6d, 0x25,
		},
	},
	{
		.algorithm = NID_sha3_384,
		.in = "",
		.in_len = 0,
		.out = {
			0x0c, 0x63, 0xa7, 0x5b, 0x84, 0x5e, 0x4f, 0x7d,
			0x01, 0x10, 0x7d, 0x85, 0x2e, 0x4c, 0x24, 0x85,
			0xc5, 0x1a, 0x50, 0xaa, 0xaa, 0x94, 0xfc, 0x61,
			0x99, 0x5e, 0x71, 0xbb, 0xee, 0x98, 0x3a, 0x2a,
			0xc3, 0x71, 0x38, 0x31, 0x26, 0x4a, 0xdb, 0x47,
			0xfb, 0x6b, 0xd1, 0xe0, 0x58, 0xd5, 0xf0, 0x04,
		},
	},
	{
		.algorithm = NID_sha3_384,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x99, 0x1c, 0x66, 0x57, 0x55, 0xeb, 0x3a, 0x4b,
			0x6b, 0xbd, 0xfb, 0x75, 0xc7, 0x8a, 0x49, 0x2e,
			0x8c, 0x56, 0xa2, 0x2c, 0x5c, 0x4d, 0x7e, 0x42,
			0x9b, 0xfd, 0xbc, 0x32, 0xb9, 0xd4, 0xad, 0x5a,
			0xa0, 0x4a, 0x1f, 0x07, 0x6e, 0x62, 0xfe, 0xa1,
			0x9e, 0xef, 0x51, 0xac, 0xd0, 0x65, 0x7c, 0x22,
		},
	},
	{
		.algorithm = NID_sha3_384,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0x79, 0x40, 0x7d, 0x3b, 0x59, 0x16, 0xb5, 0x9c,
			0x3e, 0x30, 0xb0, 0x98, 0x22, 0x97, 0x47, 0x91,
			0xc3, 0x13, 0xfb, 0x9e, 0xcc, 0x84, 0x9e, 0x40,
			0x6f, 0x23, 0x59, 0x2d, 0x04, 0xf6, 0x25, 0xdc,
			0x8c, 0x70, 0x9b, 0x98, 0xb4, 0x3b, 0x38, 0x52,
			0xb3, 0x37, 0x21, 0x61, 0x79, 0xaa, 0x7f, 0xc7,
		},
	},

	/* SHA3-512 */
	{
		.algorithm = NID_sha3_512,
		.in = "abc",
		.in_len = 3,
		.out = {
			0xb7, 0x51, 0x85, 0x0b, 0x1a, 0x57, 0x16, 0x8a,
			0x56, 0x93, 0xcd, 0x92, 0x4b, 0x6b, 0x09, 0x6e,
			0x08, 0xf6, 0x21, 0x82, 0x74, 0x44, 0xf7, 0x0d,
			0x88, 0x4f, 0x5d, 0x02, 0x40, 0xd2, 0x71, 0x2e,
			0x10, 0xe1, 0x16, 0xe9, 0x19, 0x2a, 0xf3, 0xc9,
			0x1a, 0x7e, 0xc5, 0x76, 0x47, 0xe3, 0x93, 0x40,
			0x57, 0x34, 0x0b, 0x4c, 0xf4, 0x08, 0xd5, 0xa5,
			0x65, 0x92, 0xf8, 0x27, 0x4e, 0xec, 0x53, 0xf0,
		},
	},
	{
		.algorithm = NID_sha3_512,
		.in = "",
		.in_len = 0,
		.out = {
			0xa6, 0x9f, 0x73, 0xcc, 0xa2, 0x3a, 0x9a, 0xc5,
			0xc8, 0xb5, 0x67, 0xdc, 0x18, 0x5a, 0x75, 0x6e,
			0x97, 0xc9, 0x82, 0x16, 0x4f, 0xe2, 0x58, 0x59,
			0xe0, 0xd1, 0xdc, 0xc1, 0x47, 0x5c, 0x80, 0xa6,
			0x15, 0xb2, 0x12, 0x3a, 0xf1, 0xf5, 0xf9, 0x4c,
			0x11, 0xe3, 0xe9, 0x40, 0x2c, 0x3a, 0xc5, 0x58,
			0xf5, 0x00, 0x19, 0x9d, 0x95, 0xb6, 0xd3, 0xe3,
			0x01, 0x75, 0x85, 0x86, 0x28, 0x1d, 0xcd, 0x26,
		},
	},
	{
		.algorithm = NID_sha3_512,
		.in =
		    "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmno"
		    "mnopnopq",
		.in_len = 56,
		.out = {
			0x04, 0xa3, 0x71, 0xe8, 0x4e, 0xcf, 0xb5, 0xb8,
			0xb7, 0x7c, 0xb4, 0x86, 0x10, 0xfc, 0xa8, 0x18,
			0x2d, 0xd4, 0x57, 0xce, 0x6f, 0x32, 0x6a, 0x0f,
			0xd3, 0xd7, 0xec, 0x2f, 0x1e, 0x91, 0x63, 0x6d,
			0xee, 0x69, 0x1f, 0xbe, 0x0c, 0x98, 0x53, 0x02,
			0xba, 0x1b, 0x0d, 0x8d, 0xc7, 0x8c, 0x08, 0x63,
			0x46, 0xb5, 0x33, 0xb4, 0x9c, 0x03, 0x0d, 0x99,
			0xa2, 0x7d, 0xaf, 0x11, 0x39, 0xd6, 0xe7, 0x5e,
		},
	},
	{
		.algorithm = NID_sha3_512,
		.in =
		    "abcdefghbcdefghicdefghijdefghijkefghijklfghijklm"
		    "ghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrs"
		    "mnopqrstnopqrstu",
		.in_len = 112,
		.out = {
			0xaf, 0xeb, 0xb2, 0xef, 0x54, 0x2e, 0x65, 0x79,
			0xc5, 0x0c, 0xad, 0x06, 0xd2, 0xe5, 0x78, 0xf9,
			0xf8, 0xdd, 0x68, 0x81, 0xd7, 0xdc, 0x82, 0x4d,
			0x26, 0x36, 0x0f, 0xee, 0xbf, 0x18, 0xa4, 0xfa,
			0x73, 0xe3, 0x26, 0x11, 0x22, 0x94, 0x8e, 0xfc,
			0xfd, 0x49, 0x2e, 0x74, 0xe8, 0x2e, 0x21, 0x89,
			0xed, 0x0f, 0xb4, 0x40, 0xd1, 0x87, 0xf3, 0x82,
			0x27, 0x0c, 0xb4, 0x55, 0xf2, 0x1d, 0xd1, 0x85,
		},
	},
};

struct sha_repetition_test {
	const int algorithm;
	const uint8_t in;
	const size_t in_repetitions;
	const uint8_t out[EVP_MAX_MD_SIZE];
};

static const struct sha_repetition_test sha_repetition_tests[] = {
	/* SHA-1 */
	{
		.algorithm = NID_sha1,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0x34, 0xaa, 0x97, 0x3c, 0xd4, 0xc4, 0xda, 0xa4,
			0xf6, 0x1e, 0xeb, 0x2b, 0xdb, 0xad, 0x27, 0x31,
			0x65, 0x34, 0x01, 0x6f,
		},
	},

	/* SHA-224 */
	{
		.algorithm = NID_sha224,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0x20, 0x79, 0x46, 0x55, 0x98, 0x0c, 0x91, 0xd8,
			0xbb, 0xb4, 0xc1, 0xea, 0x97, 0x61, 0x8a, 0x4b,
			0xf0, 0x3f, 0x42, 0x58, 0x19, 0x48, 0xb2, 0xee,
			0x4e, 0xe7, 0xad, 0x67,
		},
	},

	/* SHA-256 */
	{
		.algorithm = NID_sha256,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0xcd, 0xc7, 0x6e, 0x5c, 0x99, 0x14, 0xfb, 0x92,
			0x81, 0xa1, 0xc7, 0xe2, 0x84, 0xd7, 0x3e, 0x67,
			0xf1, 0x80, 0x9a, 0x48, 0xa4, 0x97, 0x20, 0x0e,
			0x04, 0x6d, 0x39, 0xcc, 0xc7, 0x11, 0x2c, 0xd0,
		},
	},

	/* SHA-384 */
	{
		.algorithm = NID_sha384,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0x9d, 0x0e, 0x18, 0x09, 0x71, 0x64, 0x74, 0xcb,
			0x08, 0x6e, 0x83, 0x4e, 0x31, 0x0a, 0x4a, 0x1c,
			0xed, 0x14, 0x9e, 0x9c, 0x00, 0xf2, 0x48, 0x52,
			0x79, 0x72, 0xce, 0xc5, 0x70, 0x4c, 0x2a, 0x5b,
			0x07, 0xb8, 0xb3, 0xdc, 0x38, 0xec, 0xc4, 0xeb,
			0xae, 0x97, 0xdd, 0xd8, 0x7f, 0x3d, 0x89, 0x85,
		},
	},

	/* SHA-512 */
	{
		.algorithm = NID_sha512,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0xe7, 0x18, 0x48, 0x3d, 0x0c, 0xe7, 0x69, 0x64,
			0x4e, 0x2e, 0x42, 0xc7, 0xbc, 0x15, 0xb4, 0x63,
			0x8e, 0x1f, 0x98, 0xb1, 0x3b, 0x20, 0x44, 0x28,
			0x56, 0x32, 0xa8, 0x03, 0xaf, 0xa9, 0x73, 0xeb,
			0xde, 0x0f, 0xf2, 0x44, 0x87, 0x7e, 0xa6, 0x0a,
			0x4c, 0xb0, 0x43, 0x2c, 0xe5, 0x77, 0xc3, 0x1b,
			0xeb, 0x00, 0x9c, 0x5c, 0x2c, 0x49, 0xaa, 0x2e,
			0x4e, 0xad, 0xb2, 0x17, 0xad, 0x8c, 0xc0, 0x9b,
		},
	},

	/* SHA3-224 */
	{
		.algorithm = NID_sha3_224,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0xd6, 0x93, 0x35, 0xb9, 0x33, 0x25, 0x19, 0x2e,
			0x51, 0x6a, 0x91, 0x2e, 0x6d, 0x19, 0xa1, 0x5c,
			0xb5, 0x1c, 0x6e, 0xd5, 0xc1, 0x52, 0x43, 0xe7,
			0xa7, 0xfd, 0x65, 0x3c,
		},
	},

	/* SHA3-256 */
	{
		.algorithm = NID_sha3_256,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0x5c, 0x88, 0x75, 0xae, 0x47, 0x4a, 0x36, 0x34,
			0xba, 0x4f, 0xd5, 0x5e, 0xc8, 0x5b, 0xff, 0xd6,
			0x61, 0xf3, 0x2a, 0xca, 0x75, 0xc6, 0xd6, 0x99,
			0xd0, 0xcd, 0xcb, 0x6c, 0x11, 0x58, 0x91, 0xc1,
		},
	},

	/* SHA3-384 */
	{
		.algorithm = NID_sha3_384,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0xee, 0xe9, 0xe2, 0x4d, 0x78, 0xc1, 0x85, 0x53,
			0x37, 0x98, 0x34, 0x51, 0xdf, 0x97, 0xc8, 0xad,
			0x9e, 0xed, 0xf2, 0x56, 0xc6, 0x33, 0x4f, 0x8e,
			0x94, 0x8d, 0x25, 0x2d, 0x5e, 0x0e, 0x76, 0x84,
			0x7a, 0xa0, 0x77, 0x4d, 0xdb, 0x90, 0xa8, 0x42,
			0x19, 0x0d, 0x2c, 0x55, 0x8b, 0x4b, 0x83, 0x40,
		},
	},

	/* SHA3-512 */
	{
		.algorithm = NID_sha3_512,
		.in = 'a',
		.in_repetitions = 1000000,
		.out = {
			0x3c, 0x3a, 0x87, 0x6d, 0xa1, 0x40, 0x34, 0xab,
			0x60, 0x62, 0x7c, 0x07, 0x7b, 0xb9, 0x8f, 0x7e,
			0x12, 0x0a, 0x2a, 0x53, 0x70, 0x21, 0x2d, 0xff,
			0xb3, 0x38, 0x5a, 0x18, 0xd4, 0xf3, 0x88, 0x59,
			0xed, 0x31, 0x1d, 0x0a, 0x9d, 0x51, 0x41, 0xce,
			0x9c, 0xc5, 0xc6, 0x6e, 0xe6, 0x89, 0xb2, 0x66,
			0xa8, 0xaa, 0x18, 0xac, 0xe8, 0x28, 0x2a, 0x0e,
			0x0d, 0xb5, 0x96, 0xc9, 0x0b, 0x0a, 0x7b, 0x87,
		},
	},
};

#define N_SHA_TESTS (sizeof(sha_tests) / sizeof(sha_tests[0]))
#define N_SHA_REPETITION_TESTS (sizeof(sha_repetition_tests) / sizeof(sha_repetition_tests[0]))

typedef unsigned char *(*sha_hash_func)(const unsigned char *, size_t,
    unsigned char *);

static int
sha_hash_from_algorithm(int algorithm, const char **out_label,
    sha_hash_func *out_func, const EVP_MD **out_md, size_t *out_len)
{
	const char *label;
	sha_hash_func sha_func;
	const EVP_MD *md;
	size_t len;

	switch (algorithm) {
	case NID_sha1:
		label = SN_sha1;
		sha_func = SHA1;
		md = EVP_sha1();
		len = SHA_DIGEST_LENGTH;
		break;
	case NID_sha224:
		label = SN_sha224;
		sha_func = SHA224;
		md = EVP_sha224();
		len = SHA224_DIGEST_LENGTH;
		break;
	case NID_sha256:
		label = SN_sha256;
		sha_func = SHA256;
		md = EVP_sha256();
		len = SHA256_DIGEST_LENGTH;
		break;
	case NID_sha384:
		label = SN_sha384;
		sha_func = SHA384;
		md = EVP_sha384();
		len = SHA384_DIGEST_LENGTH;
		break;
	case NID_sha512:
		label = SN_sha512;
		sha_func = SHA512;
		md = EVP_sha512();
		len = SHA512_DIGEST_LENGTH;
		break;
	case NID_sha3_224:
		label = SN_sha3_224;
		sha_func = NULL;
		md = EVP_sha3_224();
		len = 224 / 8;
		break;
	case NID_sha3_256:
		label = SN_sha3_256;
		sha_func = NULL;
		md = EVP_sha3_256();
		len = 256 / 8;
		break;
	case NID_sha3_384:
		label = SN_sha3_384;
		sha_func = NULL;
		md = EVP_sha3_384();
		len = 384 / 8;
		break;
	case NID_sha3_512:
		label = SN_sha3_512;
		sha_func = NULL;
		md = EVP_sha3_512();
		len = 512 / 8;
		break;
	default:
		fprintf(stderr, "FAIL: unknown algorithm (%d)\n",
		    algorithm);
		return 0;
	}

	if (out_label != NULL)
		*out_label = label;
	if (out_func != NULL)
		*out_func = sha_func;
	if (out_md != NULL)
		*out_md = md;
	if (out_len != NULL)
		*out_len = len;

	return 1;
}

static int
sha_test(void)
{
	sha_hash_func sha_func;
	const struct sha_test *st;
	EVP_MD_CTX *hash = NULL;
	const EVP_MD *md;
	uint8_t out[EVP_MAX_MD_SIZE];
	size_t in_len, out_len;
	size_t i;
	const char *label;
	int failed = 1;

	if ((hash = EVP_MD_CTX_new()) == NULL) {
		fprintf(stderr, "FAIL: EVP_MD_CTX_new() failed\n");
		goto failed;
	}

	for (i = 0; i < N_SHA_TESTS; i++) {
		st = &sha_tests[i];
		if (!sha_hash_from_algorithm(st->algorithm, &label, &sha_func,
		    &md, &out_len))
			goto failed;

		/* Digest */
		if (sha_func != NULL) {
			memset(out, 0, sizeof(out));
			sha_func(st->in, st->in_len, out);
			if (memcmp(st->out, out, out_len) != 0) {
				fprintf(stderr, "FAIL (%s:%zu): mismatch\n",
				    label, i);
				goto failed;
			}
		}

		/* EVP single-shot digest */
		memset(out, 0, sizeof(out));
		if (!EVP_Digest(st->in, st->in_len, out, NULL, md, NULL)) {
			fprintf(stderr, "FAIL (%s:%zu): EVP_Digest failed\n",
			    label, i);
			goto failed;
		}

		if (memcmp(st->out, out, out_len) != 0) {
			fprintf(stderr,
			    "FAIL (%s:%zu): EVP single-shot mismatch\n",
			    label, i);
			goto failed;
		}

		/* EVP digest */
		memset(out, 0, sizeof(out));
		if (!EVP_DigestInit_ex(hash, md, NULL)) {
			fprintf(stderr,
			    "FAIL (%s:%zu): EVP_DigestInit_ex failed\n",
			    label, i);
			goto failed;
		}

		in_len = st->in_len / 2;
		if (!EVP_DigestUpdate(hash, st->in, in_len)) {
			fprintf(stderr,
			    "FAIL (%s:%zu): EVP_DigestUpdate first half "
			    "failed\n", label, i);
			goto failed;
		}

		if (!EVP_DigestUpdate(hash, st->in + in_len,
		    st->in_len - in_len)) {
			fprintf(stderr,
			    "FAIL (%s:%zu): EVP_DigestUpdate second half "
			    "failed\n", label, i);
			goto failed;
		}

		if (!EVP_DigestFinal_ex(hash, out, NULL)) {
			fprintf(stderr,
			    "FAIL (%s:%zu): EVP_DigestFinal_ex failed\n",
			    label, i);
			goto failed;
		}

		if (memcmp(st->out, out, out_len) != 0) {
			fprintf(stderr, "FAIL (%s:%zu): EVP mismatch\n",
			    label, i);
			goto failed;
		}
	}

	failed = 0;

 failed:
	EVP_MD_CTX_free(hash);
	return failed;
}

static int
sha_repetition_test(void)
{
	const struct sha_repetition_test *st;
	EVP_MD_CTX *hash = NULL;
	const EVP_MD *md;
	uint8_t buf[1024];
	uint8_t out[EVP_MAX_MD_SIZE];
	size_t out_len, part_len;
	size_t i, j;
	const char *label;
	int failed = 1;

	if ((hash = EVP_MD_CTX_new()) == NULL) {
		fprintf(stderr, "FAIL: EVP_MD_CTX_new() failed\n");
		goto failed;
	}

	for (i = 0; i < N_SHA_REPETITION_TESTS; i++) {
		st = &sha_repetition_tests[i];
		if (!sha_hash_from_algorithm(st->algorithm, &label, NULL, &md,
		    &out_len))
			goto failed;

		/* EVP digest */
		if (!EVP_DigestInit_ex(hash, md, NULL)) {
			fprintf(stderr,
			    "FAIL (%s:%zu): EVP_DigestInit_ex failed\n",
			    label, i);
			goto failed;
		}

		memset(buf, st->in, sizeof(buf));

		for (j = 0; j < st->in_repetitions;) {
			part_len = arc4random_uniform(sizeof(buf));
			if (part_len > st->in_repetitions - j)
				part_len = st->in_repetitions - j;

			if (!EVP_DigestUpdate(hash, buf, part_len)) {
				fprintf(stderr,
				    "FAIL (%s:%zu): EVP_DigestUpdate failed\n",
				    label, i);
				goto failed;
			}

			j += part_len;
		}

		if (!EVP_DigestFinal_ex(hash, out, NULL)) {
			fprintf(stderr,
			    "FAIL (%s:%zu): EVP_DigestFinal_ex failed\n",
			    label, i);
			goto failed;
		}

		if (memcmp(st->out, out, out_len) != 0) {
			fprintf(stderr, "FAIL (%s:%zu): EVP mismatch\n",
			    label, i);
			goto failed;
		}
	}

	failed = 0;

 failed:
	EVP_MD_CTX_free(hash);
	return failed;
}

int
main(int argc, char **argv)
{
	int failed = 0;

	failed |= sha_test();
	failed |= sha_repetition_test();

	return failed;
}
